/******************************************************************************
	CTextDocument.c
	

				CTextDocument Class
	

	Copyright (C) 1985-1992  New York University
	Copyright (C) 1994 George Washington University
	 
	This file is part of the GWAdaEd system, an extension of the Ada/Ed-C
	system.  See the Ada/Ed README file for warranty (none) and distribution
	info and also the GNU General Public License for more details.


 ******************************************************************************/

#include <Global.h>
#include <Commands.h>
#include <CApplication.h>
#include <CBartender.h>
#include <CDataFile.h>
#include <CResFile.h>
#include <CDecorator.h>
#include <CDesktop.h>
#include <CError.h>
#include <CPanorama.h>
#include <CScrollPane.h>
#include <TBUtilities.h>
#include <CWindow.h>


#include "CTextDocument.h"
#include "CTextPane.h"
#include "FindWindowSize.h"

#include "CAdaApp.h"
#include "CGotoLine.h"
#include "CFind.h"

#include "AppCommands.h"
#include "SFGetFolder.h"

#define WindRsrcType	'DOC1'
#define WindRsrcID		500		/* Window resource data */

#define FontRsrcType	'DOC2'
#define FontRsrcID		500		/* Font resource data */

/**** Font Defaults ****/
#define kDefaultFont	monaco
#define kDefaultSize	9

extern	CApplication *gApplication;	/* The application */
extern	CBartender	*gBartender;	/* The menu handling object */
extern	CDecorator	*gDecorator;	/* Window dressing object	*/
extern	CDesktop	*gDesktop;		/* The enclosure for all windows */
extern	OSType		gSignature;		/* The application's signature */

/***
 * ITextDocument
 *
 *	Call the default method to initialize whatever is needed.
 *	All text documents in GW Ada are printable.
 *
 ***/
 
void	CTextDocument::ITextDocument(CApplication *aSupervisor)
{
	CDocument::IDocument(aSupervisor, true);
	fReadOnly = true;

	fFont = kDefaultFont;			// defaults
	fSize = kDefaultSize;
	fHaveWindSize = false;
}

/***
 * Notify
 *
 *	This calls the inherited method to make the work, and
 *	then resets the dirty flag if the document is readonly.
 *
 ***/

void	CTextDocument::Notify(CTask *theTask)
{
	inherited::Notify(theTask);
	if (IsItReadOnly())
		dirty = FALSE;		// don't let it be dirty
}

/***
 *	IsItReadOnly
 *
 *	Return value in 'fReadOnly'
 *
 ***/

Boolean	CTextDocument::IsItReadOnly(void)
{
	return fReadOnly;
}

/***
 * GetSelectionStr
 *
 *	Ask the text pane for the contents of the selection.  Then
 *  do some simple text processing to remove spaces, new lines
 *  and a few others.
 *
 ***/

void	CTextDocument::GetSelectionStr(Str255 sel)
{
	long selStart, selEnd, hSize;
	Handle text;
	int i;

	fTextEdit->GetSelection(&selStart, &selEnd);
	text = fTextEdit->CopyTextRange(selStart, selEnd);
	hSize = GetHandleSize(text);
	if (hSize < 255) {
		MoveHHi(text);
		HLock(text);
		BlockMove(*text, &(sel[1]), hSize);
		sel[0] = hSize;
		HUnlock(text);
		DisposeHandle(text);
	}
	else {
		DisposeHandle(text);
		sel[0] = 0;
	}

	// make sure there is no end-of-line in this string
	for (i = 1; i <= sel[0]; i++) {
		if (sel[i] == '\r' || sel[i] == '\n') {
			// if we find end-of-line, then cut
			// the string short
			sel[0] = i-1;
			break;
		}
	}
}

/***
 * GetNumLines
 *
 *	Ask the fTextEdit for the number of lines stored there
 *
 ***/

long	CTextDocument::GetNumLines(void)
{
	return fTextEdit->GetNumLines();
}

/***
 *	GetFileName
 *
 *  Return the name of the file associated with this document.
 *	If the document haven't been saved, return an empty Pascal
 *	string.
 *
 ***/

void	CTextDocument::GetFileName(Str255 fileName)
{
	if (itsFile)
		itsFile->GetName(fileName);
	else
		fileName[0] = 0;
}

/***
 *	GetFilePath
 *
 *	Similar to GetFileName() but returns the path to this document.
 *
 ***/

void	CTextDocument::GetFilePath(Str255 filePath)
{
	if (itsFile) {
		FSSpec aFileSpec;
		Str255 justTheName, path;
		int i;

		itsFile->GetFSSpec(&aFileSpec);
		CopyPString(aFileSpec.name, justTheName);
		GetFullPath(aFileSpec, &path);

		// now cut out the file name from the path
		path[0] = path[0] - justTheName[0];
		CopyPString(path, filePath);
	}
	else
		filePath[0] = 0;
}

/***
 *	GetFSSpec(FSSpec *spec)
 *
 *	Returns the file spec from itsFile.
 *
 ***/

void	CTextDocument::GetFSSpec(FSSpec *spec)
{
	if (itsFile)
		itsFile->GetFSSpec(spec);
	else
		spec->name[0] = 0;
}

/***
 * NewFile
 *
 *	When the user chooses New from the File menu, the CreateDocument()
 *	method in your Application class will send a newly created document
 *	this message. This method needs to create a new window, ready to
 *	work on a new document.
 *
 *	Since this method and the OpenFile() method share the code for creating
 *	the window, you should use an auxiliary window-building method.
 *
 ***/
void CTextDocument::NewFile(void)
{
	BuildWindow(NULL);		// no data to display
	itsWindow->Select();
}


/***
 * OpenFile
 *
 *	When the user chooses Open from the File menu, the OpenDocument()
 *	method in your Application class will let the user choose a file
 *	and then send a newly created document this message. The information
 *	about the file is in the SFReply record.
 *
 *	In this method, you need to open the file and display its contents
 *	in a window. This method uses the auxiliary window-building method.
 *
 ***/

void CTextDocument::OpenFile(SFReply *macSFReply)

{
	CDataFile	*theFile;
	Handle		theData;
	Str63		theName;
	OSErr		theError;


	ReadRsrcFork(macSFReply);		// read rsrc fork first

		/**
		 **	Be sure to set the instance variable
		 **	so other methods can use the file if they
		 **	need to. This is especially important if
		 **	you leave the file open in this method.
		 **	If you close the file after reading it, you
		 **	should be sure to set itsFile to NULL.
		 **
		 **/
	theFile = new(CDataFile);
	itsFile = theFile;

	theFile->IDataFile();
	theFile->SFSpecify(macSFReply);
	

		/**
		 **	Send the file an Open() message to
		 **	open it. You can use the ReadSome() or
		 **	ReadAll() methods to get the contents of the file.
		 **
		 **/

	theFile->Open(fsRdWrPerm);
	
	if (theFile->GetLength() > 32000L)
	{
		ParamText( "\pCan't open a file bigger than 32K.", "\p", "\p", "\p");
		PositionDialog('ALRT', 128);
		InitCursor();
		Alert(128, NULL);
		
		Dispose();
		return;
	}
	

    theData = theFile->ReadAll();     /* ReadAll() creates the handle */

	BuildWindow(theData);
	DisposHandle(theData);

		/**
		 **	In this implementation, we leave the file
		 **	open. You might want to close it after
		 **	you've read in all the data.
		 **
		 **/

	itsFile->GetName(theName);
	itsWindow->SetTitle(theName);
	itsWindow->Select();			/* Don't forget to make the window active */
}

/***
 * ReadRsrcFork
 *
 *	Read this documents resource fork to find the default
 *	font, font size, and window size.  If the document doesn't
 *	have a resource fork, then ignore it.  The default values
 *	for font/size/window are set in the initialize method of
 *	this class.
 *
 ***/

void	CTextDocument::ReadRsrcFork(SFReply *macSFReply)
{
	CResFile	*itsResFile = NULL;
	Handle fHdl = NULL;
	Handle wHdl = NULL;

	TRY {
		// Now check resource file to see if it has font information
		itsResFile = new CResFile;
		itsResFile->IResFile();
		itsResFile->SFSpecify(macSFReply);
	
		// if the file has a resource fork, then try to read
		// the resources.  If it doesn't have a resource fork
		// then ignore the resources (i.e. do nothing)

		if (itsResFile->HasResFork()) {
			short fontInfo[2];
			Rect windSize;

			itsResFile->Open(fsRdWrPerm);
			itsResFile->MakeCurrent();
			
			// Get Resource for Font Information
			fHdl = Get1Resource(FontRsrcType, FontRsrcID);
			if (fHdl && GetHandleSize(fHdl) == sizeof(fontInfo)) {

				HLock(fHdl);
				BlockMove(*fHdl, &fontInfo, sizeof(fontInfo));
				HUnlock(fHdl);
				
				fFont = fontInfo[0];		// copy data
				fSize = fontInfo[1];
			}
			ForgetResource(fHdl);

			// Get Resource that contains window information
			wHdl = Get1Resource(WindRsrcType, WindRsrcID);
			if (wHdl && GetHandleSize(wHdl) == sizeof(windSize)) {

				HLock(wHdl);
				BlockMove(*wHdl, &windSize, sizeof(windSize));
				HUnlock(wHdl);
				
				fWindSizeLoc = windSize;	// copy data
				fHaveWindSize = true;
			}
			ForgetResource(wHdl);


			// Close resource file and release object
			itsResFile->Close();
			ForgetObject(itsResFile);
		}
	}
	CATCH {
		ForgetObject(itsResFile);
		ForgetResource(fHdl);
		ForgetResource(wHdl);
	}
	ENDTRY;
}

/***
 * WriteRsrcFork
 *
 *	Write resources to the resource fork of this document.
 *	The resource fork is not maintained opened, so
 *	we must reopen it here, delete existing resources
 *	and then add the resources
 *
 ***/

void	CTextDocument::WriteRsrcFork(void)
{
	CResFile	*itsResFile = NULL;
	Handle		 wHdl = NULL;
	Handle		 fHdl = NULL;
	short	fontInfo[2];
	Rect	windSize;
	FSSpec	spec;

	TRY {
		itsFile->GetFSSpec(&spec);		// get spec from data file
		itsResFile = new(CResFile);
		itsResFile->IResFile();
		itsResFile->SpecifyFSSpec(&spec);	// and use it for res file
		if (!itsResFile->HasResFork())
			itsResFile->CreateNew(gSignature, 'TEXT');

		itsResFile->Open(fsRdWrPerm);
		itsResFile->MakeCurrent();
				
		// This is the information that we will save to the
		// resource file...

		// get the font/size from the text edit field
		//fFont = (**fTextEdit->macTE).txFont;
		//fSize = (**fTextEdit->macTE).txSize;
		fFont = fTextEdit->GetFontNum();
		fSize = fTextEdit->GetFontSize();

		fontInfo[0] = fFont;
		fontInfo[1] = fSize;

	
		// ... see if there is already a resource with the
		// right id and type...
		fHdl = Get1Resource(FontRsrcType, FontRsrcID);
		if (fHdl == NULL) {

			// Resource not there, so add a new one
			fHdl = NewHandle(sizeof(fontInfo));
			HLock(fHdl);
			BlockMove(&fontInfo, *fHdl, sizeof(fontInfo));
			HUnlock(fHdl);

			AddResource(fHdl, FontRsrcType, FontRsrcID, "\pFont");
			FailResError();

			// what to do with the handle allocated by NewHandle?
			// is it part of the resource file?
		}
	
	
		else if (GetHandleSize(fHdl) == sizeof(fontInfo)) {
	
			// Resource there and same size as our data, so just
			// update the values and mark the resource as changed

			HLock(fHdl);
			BlockMove(&fontInfo, *fHdl, sizeof(fontInfo));
			HUnlock(fHdl);

			ChangedResource(fHdl);
			FailResError();
		}
	
		else {
	
			// Resource there, but not same size as current font
			// data.  Delete resource, and add a new one.
			RmveResource(fHdl);
			FailResError();

			SetHandleSize(fHdl, sizeof(fontInfo));
			FailMemError();

			HLock(fHdl);
			BlockMove(&fontInfo, *fHdl, sizeof(fontInfo));
			HUnlock(fHdl);

			AddResource(fHdl, FontRsrcType, FontRsrcID, "\pFont");
			FailResError();
		}
	
	
		/*** Finished writting the font information, now follow
		 *** a similar approach for the window information.
		 ***/

		// Get values that will be written to the resource
		FindWindowSize(itsWindow, &windSize);
	
		// Get Resource that contains window information
		wHdl = Get1Resource(WindRsrcType, WindRsrcID);
		if (wHdl == NULL) {
	
			// Resource not there, so add a new one
			wHdl = NewHandle(sizeof(windSize));
			HLock(wHdl);
			BlockMove(&windSize, *wHdl, sizeof(windSize));
			HUnlock(wHdl);

			AddResource(wHdl, WindRsrcType, WindRsrcID, "\pWindow");
			FailResError();
		}
	
		else if (GetHandleSize(wHdl) == sizeof(windSize)) {
	
			// Resource there and same size as our data, so just
			// update the values and mark the resource as changed
			HLock(wHdl);
			BlockMove(&windSize, *wHdl, sizeof(windSize));
			HUnlock(wHdl);

			ChangedResource(wHdl);
			FailResError();
		}
	
		else {
	
			// Resource there, but not same size as current font
			// data.  Delete resource, and add a new one.
			RmveResource(wHdl);
	
			SetHandleSize(wHdl, sizeof(windSize));
			FailMemError();

			HLock(wHdl);
			BlockMove(&windSize, *wHdl, sizeof(windSize));
			HUnlock(wHdl);

			AddResource(wHdl, WindRsrcType, WindRsrcID, "\pWindow");
			FailResError();
		}
	
	
	
		// This will cause an update.
		itsResFile->Close();
		ForgetObject(itsResFile);

	}
	CATCH {
		ForgetResource(wHdl);
		ForgetResource(fHdl);
		ForgetObject(itsResFile);
	}
	ENDTRY;
}



/***
 * DoSave
 *
 *	This method handles what happens when the user chooses Save from the
 *	File menu. This method should return TRUE if the file save was successful.
 *	If there is no file associated with the document, you should send a
 *	DoSaveFileAs() message.
 *
 ***/

Boolean CTextDocument::DoSave(void)

{
	Handle		theData;

	if (itsFile == NULL)
		return(DoSaveFileAs());

	else {
		//theData = (**(fTextEdit->macTE)).hText;
		theData = fTextEdit->GetTextHandle();
		((CDataFile *)itsFile)->WriteAll(theData);
		dirty = FALSE;		/* Document is no longer dirty		*/

		WriteRsrcFork();

		gBartender->DisableCmd(cmdSave);
		return(TRUE);		/* Save was successful				*/
	}
}


/***
 * DoSaveAs
 *
 *	This method handles what happens when the user chooses Save As from
 *	File menu. The default DoCommand() method for documents sends a DoSaveFileAs()
 *	message which displays a standard put file dialog and sends this message.
 *	The SFReply record contains all the information about the file you're about
 *	to create.
 *
 ***/

Boolean CTextDocument::DoSaveAs(SFReply *macSFReply)

{
		/**
		 **	If there's a file associated with this document
		 **	already, close it. The Dispose() method for files
		 **	sends a Close() message to the file before releasing
		 **	its memory.
		 **
		 **/
		 
	if (itsFile != NULL)
		itsFile->Dispose();


		/**
		 **	Create a new file, and then save it normally.
		 **
		 **/

	itsFile = new(CDataFile);
	((CDataFile *)itsFile)->IDataFile();
	itsFile->SFSpecify(macSFReply);
	if (itsFile->ExistsOnDisk())	/* If file is on disk, throw it out	*/
		itsFile->ThrowOut();
	itsFile->CreateNew(gSignature, 'TEXT');
	itsFile->Open(fsRdWrPerm);

	itsWindow->SetTitle(macSFReply->fName);

	return( DoSave() );
}


/***
 * BuildWindow
 *
 *	This is the auxiliary window-building method that the
 *	NewFile() and OpenFile() methods use to create a window.
 *
 *	In this implementation, the argument is the data to display.
 *
 ***/

void CTextDocument::BuildWindow (Handle theData)

{
	MakeWindow();
	BuildWindowContents();

	if (theData)
		fTextEdit->SetTextHandle(theData);

	// set the font/size into the text edit field
	fTextEdit->SetFontNumber(fFont);
	fTextEdit->SetFontSize(fSize);

	// After reading the document, change TextEdit's flag
	// to read only if needed. 
	if (IsItReadOnly())
		fTextEdit->Specify(kNotEditable, kSelectable, kStylable);

	fTextEdit->SetSelection(0, 0, false);
	fTextEdit->ScrollToSelection();

	if (fHaveWindSize) {

		// we have window size/location information, now check that
		// it is valid for this monitor...  if invalid just call
		// gDecorator...
		
		if (fWindSizeLoc.left < screenBits.bounds.left ||
			fWindSizeLoc.top < screenBits.bounds.top ||
			fWindSizeLoc.right > screenBits.bounds.right ||
			fWindSizeLoc.bottom > screenBits.bounds.bottom)
			
			gDecorator->PlaceNewWindow(itsWindow);
		else {
			itsWindow->Move(fWindSizeLoc.left, fWindSizeLoc.top);
			itsWindow->ChangeSize(fWindSizeLoc.right, fWindSizeLoc.bottom);
		}
	}
	else {
		gDecorator->PlaceNewWindow(itsWindow);
	}
}


void	CTextDocument::MakeWindow(void)
{
	itsWindow = new(CWindow);
	itsWindow->IWindow(WINDculture, FALSE, gDesktop, this);
}

/***
 *	BuildWindowContents
 *
 *	Separated from BuildWindow so that subclass can create
 *	different configurations of the contents of the window
 *
 ***/

void	CTextDocument::BuildWindowContents(void)
{
	CScrollPane		*theScrollPane;
	CTextPane		*theMainPane;
	LongPt			pt;

	theScrollPane = new(CScrollPane);
	theScrollPane->IScrollPane(itsWindow, this, 10, 10, 0, 0,
								sizELASTIC, sizELASTIC,
								TRUE, TRUE, TRUE);
	theScrollPane->FitToEnclFrame(TRUE, TRUE);

	MakeTextPane(theScrollPane);
	itsMainPane = fTextEdit;
	itsGopher = fTextEdit;

	theScrollPane->InstallPanorama(fTextEdit);
	SetLongPt(&pt, 0, 0);
	fTextEdit->ScrollTo(&pt, false);
}

void	CTextDocument::MakeTextPane(CScrollPane	*theScrollPane)
{
	Rect margin;

	fTextEdit = new(CTextPane);
	fTextEdit->ITextPane(theScrollPane, this, 0, 0, 1, 1,
		sizELASTIC, sizELASTIC);
	fTextEdit->FitToEnclosure(TRUE, TRUE);

	SetRect(&margin, 2, 2, -2, -2);
	fTextEdit->ChangeSize(&margin, FALSE);
}



/****
 *	SelectFT
 *
 *	Select area of document indicated by a pair of (line, col)
 *
 ****/

void	CTextDocument::SelectFT(long line1, long col1,
								  long line2, long col2)
{
	fTextEdit->SelectFT(line1, col1, line2, col2);
}

/****
 *	SelectLine
 *
 *	Select one whole source line
 *
 ****/

void	CTextDocument::SelectLine(long line)
{
	fTextEdit->SelectLine(line);
}

void	CTextDocument::ScrollToSelection(void)
{
	fTextEdit->ScrollToSelection();
}

void	CTextDocument::Select(void)
{
	itsWindow->Select();
}

/***
 *	GetTextHandle
 *
 *	Return handle to text record of the subview with the CEditText
 *
 ***/

Handle	CTextDocument::GetTextHandle(void)		/* don't change it! */
{
	return fTextEdit->GetTextHandle();
}


/***
 *	UpdateMenus
 *
 *	Update menu states based on what the current context of
 *	this document
 *
 ***/

		static void AbleCmd(CAdaApp *ap, long cmd)
		{
			if (ap->CanCmd(cmd))
				gBartender->EnableCmd(cmd);
			else
				gBartender->DisableCmd(cmd);
		}


void		CTextDocument::UpdateMenus(void)
{
	long selStart, selEnd, hSize, lines, cmdNo;
	Str255 sel;

	inherited::UpdateMenus();
	
	AbleCmd((CAdaApp *)gApplication, cmdFind);
	AbleCmd((CAdaApp *)gApplication, cmdFindAgain);
	
	if (!IsItReadOnly()) {
		AbleCmd((CAdaApp *)gApplication, cmdReplace);
		AbleCmd((CAdaApp *)gApplication, cmdReplaceAll);
		AbleCmd((CAdaApp *)gApplication, cmdReplaceFindAgain);
	}

	cmdNo = (-(((long)MENUedit) << 16) - 11);		// 11 = Font
	gBartender->EnableCmd(cmdNo);
	cmdNo = (-(((long)MENUedit) << 16) - 12);		// 12 = Size
	gBartender->EnableCmd(cmdNo);
	
	GetSelectionStr(sel);
	if (sel[0] > 0) {
		gBartender->EnableCmd(cmdEnterSelection);
		if (!IsItReadOnly())
			gBartender->EnableCmd(cmdEnterTemplate);
	}
	else {
		gBartender->DisableCmd(cmdEnterSelection);
		gBartender->DisableCmd(cmdEnterTemplate);
	}

	gBartender->EnableCmd(cmdGotoLine);

}



/******************************************************************************
 Clean
 
 	Set document clean.
******************************************************************************/

void	CTextDocument::Clean(void)
{
	Notify(NULL);						/* Clean up any active tasks		*/
	dirty = FALSE;						/* No longer dirty					*/
}


/***
 *	DoCommand
 *
 *	Handle default commands that apply to all text based documents
 *	in GW Ada.
 *
 ***/

void	CTextDocument::DoCommand(long theCommand)
{

CAdaApp *theApp = (CAdaApp *)gApplication;

	switch (theCommand) {
		case cmdOpenGotoLine: {
			CGotoLine	*dialog;
			long		dismissCmd;
			long		line;

			line = fTextEdit->GetCursorLine();
			dialog = new CGotoLine;
			dialog->ICGotoLine(this, fTextEdit, line);
			dialog->BeginDialog();

			dismissCmd = dialog->DoModalDialog(cmdOK);

			if (dismissCmd == cmdOK) {
				
				line = dialog->GetLineNumber();
				SelectLine(line);
				ScrollToSelection();
			}
			ForgetObject(dialog);
			break;
		}


		case cmdEnterSelection: {
			Str255 sel;

			GetSelectionStr(sel);
			theApp->SetSearchString(sel);
			break;
		}

		case cmdFind: {

			CFind		*dialog;
			long		cmd;

		  		// Respond to command by opening a dialog
		
			dialog = new CFind;
			dialog->ICFind(this);
			dialog->BeginDialog();
			
			
			// Ask the application if we should lit up the Find button
			cmd = dialog->DoModalDialog(theApp->CanCmd(cmdFindAgain) ? cmdOK : cmdDontFind);
			ForgetObject(dialog);

			if (cmd == cmdOK) {
				theApp->Search(this, cmdFind);
			}
			break;
		}


		case cmdFindAgain:
			theApp->Search(this, theCommand);
			break;


		case cmdReplace:
		case cmdReplaceFindAgain:
		case cmdReplaceAll:
			if (!IsItReadOnly())
				theApp->Replace(this, theCommand);
			break;


		case cmdEnterTemplate:
			if (!IsItReadOnly()) {
				Str255 sel;
				Handle hdl;
	
				GetSelectionStr(sel);
	
				// Search for a template with this name
				hdl = Get1NamedResource('TPL ', sel);
				ReleaseResource(hdl);
					
	
			}
			break;

		default:
			inherited::DoCommand(theCommand);
			break;
	}
}
